package data;

import java.awt.Color;
import java.io.File;
import java.util.List;

import logging.Log;
import logging.LogLevel;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

import tools.xml.XMLReader;
import tools.xml.XMLWriter;
import database.DatabaseLoginInfo;

/**
 * 
 * This class stores settings as static attributes
 * 
 * @author Martin Pabst, 2009
 * 
 */
public class DatabaseSettings {

	/**
	 * True, if database connection is ok
	 */
	private static boolean databaseConnectionOK = false;

	/**
	 * active database ("Derby" or "Postgres")
	 */
	private static String activeDatabase = "Derby";

	/**
	 * login information for derby database
	 */
	private static DatabaseLoginInfo derbyLoginInfo = new DatabaseLoginInfo();

	/**
	 * login information for postgres database
	 */
	private static DatabaseLoginInfo postgresLoginInfo = new DatabaseLoginInfo();

	/**
	 * path to settings file (holds login info for database!)
	 */
	private final static String settingsFilename = "einstellungen.xml";

	/**
	 * returns true, if database connection is ok (that means connection to asv
	 * database and to cache database.)
	 * 
	 * @return
	 */
	public static boolean isDatabaseConnectionOK() {
		return databaseConnectionOK;
	}

	/**
	 * read settings from file data/einstellungen.xml
	 */
	public static void readSettings() {
		/**
		 * XML DOM
		 */
		Document doc;

		try {

			/**
			 * read settings data form file data/einstellungen.xml
			 */
			doc = XMLReader.readFile(settingsFilename);
			Element root = XMLReader.root(doc);

			Log.outlColor("\nEinstellungen laden:", LogLevel.info, new Color(0,
					100, 0));
			// Log.outl("", LogLevel.useful);

			if (root.getNodeName().compareTo("settings") != 0) {
				throw new Exception(
						"Element \"settings\" in der Datei einstellungen.xml erwartet.");
			}

			Element settings = root;

			// Log.outl("", LogLevel.useful);

			getDatabases(settings);

			Log.outl("", LogLevel.info);

		} catch (Exception e) {
			Log.outl(e.toString() + e.getStackTrace().toString(),
					LogLevel.error);
		}

	}

	/**
	 * reads database info from given XML-Element
	 * 
	 * @param root
	 */
	private static void getDatabases(Element root) {
		Element activeDatabaseXML = XMLReader.element(root, "activedatabase");
		activeDatabase = XMLReader.attributeNotNull(activeDatabaseXML, "name");

		List<Element> databases = XMLReader.elements(root, "database");

		for (Element databaseXML : databases) {
			DatabaseLoginInfo dli = new DatabaseLoginInfo();
			dli.readFromXML(databaseXML);
			if (dli.getName().compareTo(activeDatabase) == 0) {
				dli.setAsActiveDatabase();
			}

			if (dli.getType().compareTo("derby") == 0) {
				derbyLoginInfo = dli;
			}

			if (dli.getType().compareTo("postgres") == 0) {
				postgresLoginInfo = dli;
			}
		}

		// Log.outl("", LogLevel.useful);
		Log.outColor("Verwendete Datenbank: ", LogLevel.useful, new Color(0, 0,
				255));
		Log.outl(activeDatabase, LogLevel.useful);
		Log.outl("", LogLevel.useful);

	}

	/**
	 * write settings to xml-file
	 */
	public static void writeSettings() {
		try {
			Document doc = XMLWriter.createEmptyDocument();
			Element settings = doc.createElement("settings");
			doc.appendChild(settings);

			Element adElement = XMLWriter
					.addElement("activedatabase", settings);
			XMLWriter.addAttribute(adElement, "name", activeDatabase);

			postgresLoginInfo.writeToXML(settings);

			derbyLoginInfo.writeToXML(settings);

			File file = new File(settingsFilename);
			XMLWriter.writeToFile(doc, file);

		} catch (Exception ex) {
			Log.outlColor(ex.toString(), LogLevel.error, Color.red);
		}

	}

	/**
	 * set status of database connection
	 * 
	 * @param databaseConnectionOK
	 */
	public static void setDatabaseConnectionOK(boolean databaseConnectionOK) {
		DatabaseSettings.databaseConnectionOK = databaseConnectionOK;
	}

	/**
	 * returns identifier for active databae ("Derby" or "Postgres").
	 * 
	 * @return
	 */
	public static String getActiveDatabase() {
		return activeDatabase;
	}

	/**
	 * set identifier for active databae ("Derby" or "Postgres").
	 */
	public static void setActiveDatabase(String activeDatabase) {
		DatabaseSettings.activeDatabase = activeDatabase;
	}

	/**
	 * return login info for derby database
	 * 
	 * @return
	 */
	public static DatabaseLoginInfo getDerbyLoginInfo() {
		return derbyLoginInfo;
	}

	/**
	 * set login info for derby database
	 * 
	 * @param derbyLoginInfo
	 */
	public static void setDerbyLoginInfo(DatabaseLoginInfo derbyLoginInfo) {
		DatabaseSettings.derbyLoginInfo = derbyLoginInfo;
	}

	/**
	 * get login info for postgres database
	 * 
	 * @return
	 */
	public static DatabaseLoginInfo getPostgresLoginInfo() {
		return postgresLoginInfo;
	}

	/**
	 * set login info for postgres database
	 * 
	 * @param postgresLoginInfo
	 */
	public static void setPostgresLoginInfo(DatabaseLoginInfo postgresLoginInfo) {
		DatabaseSettings.postgresLoginInfo = postgresLoginInfo;
	}

}
