package gui;

import java.awt.Color;

import javax.swing.SwingUtilities;

import logging.Log;
import logging.LogLevel;
import logging.LogOutput;
import workerthreads.ExecuteScriptsWorker;
import workerthreads.changedates.ChangeDatesWorker;
import workerthreads.fitnessescriptcorrector.FitnesseScriptCorrector;
import workerthreads.getmetadata.GetMetadataWorker;
import data.DatabaseSettings;
import data.Settings;

/**
 * 
 * An instance of this class controls the main window of the application
 * (controller of the MVC-Pattern)
 * 
 * @author Martin Pabst, 2009
 * 
 */

public class MainFrameController implements LogOutput {

	/**
	 * view of this controller == main window of application
	 */
	private MainFrame mainFrame;

	/**
	 * contructor. Instantiates MainFrame, sets log output and reads settings
	 * from einstellungen.xml
	 */

	public MainFrameController() {
		mainFrame = MainFrame.getInstance();
		mainFrame.setController(this);
		mainFrame.init();

		Log.setLogOutput(this);
		readSettings();
	}

	/**
	 * show main window
	 */
	public void showMainFrame() {

		mainFrame.setVisible(true);

	}

	/**
	 * read settings from file data/einstellungen.xml
	 */
	public void readSettings() {

		mainFrame.getTextArea().setText(""); // clear log
		mainFrame.setButtonsEnabled(false); // disable buttons

		Settings.readSettings();
		DatabaseSettings.readSettings();

		mainFrame.setButtonsEnabled(true);

	}

	/**
	 * append string to textarea on main window
	 * 
	 * @param s
	 */
	public void out(String s) {
		ColorPane ta = mainFrame.getTextArea();
		ta.append(Color.black, s);
		// ta.setCaretPosition(ta.getText().length());
	}

	/**
	 * append String to textarea on main window and begin new line
	 * 
	 * @param s
	 */
	public void outl(String s) {
		out(s + "\n");
	}

	/**
	 * append string to textarea on main window in given color
	 * 
	 * @param s
	 * @param c
	 */
	@Override
	public void outColor(String s, Color c) {
		ColorPane ta = mainFrame.getTextArea();
		ta.append(c, s);

	}

	/**
	 * append string to textarea on main window in given color and begin new
	 * line
	 * 
	 * @param s
	 * @param c
	 */
	@Override
	public void outlColor(String s, Color c) {
		outColor(s + "\n", c);
	}

	/**
	 * clear textarea in main window
	 */
	public void clearTextArea() {
		mainFrame.getTextArea().setText("");
	}

	/**
	 * returns mainframe (view of MVC-pattern)
	 * 
	 * @return
	 */
	public MainFrame getMainFrame() {
		return mainFrame;
	}

	/**
	 * enable/disable buttons on main window. Beware: buttons "compile" and
	 * "show JavaHelp" are only shown if necessary input files exist.
	 * 
	 * @param enabled
	 */
	public void setButtonsEnabled(boolean enabled) {

		Runnable doSetEnabled;

		/**
		 * we are careful and set buttons in the spring event dispatching thread
		 */
		if (enabled) {
			doSetEnabled = new Runnable() {
				public void run() {
					mainFrame.setButtonsEnabled(true);
				}
			};
		} else {
			doSetEnabled = new Runnable() {
				public void run() {
					mainFrame.setButtonsEnabled(false);
				}
			};
		}

		SwingUtilities.invokeLater(doSetEnabled);

		mainFrame.setButtonsEnabled(enabled);
	}

	/**
	 * 
	 * /** sets waitcursor (isWait == true) or default cursor (isWait == false)
	 * 
	 * @param isWait
	 */
	public void setWaitCursor(boolean isWait) {
		mainFrame.setWaitCursor(isWait);
	}

	/**
	 * setmaximum value for progressbar
	 */
	@Override
	public void setProgressBarMaximum(int n) {
		mainFrame.setProgressBarMaximum(n);

	}

	/**
	 * set value for progressbar
	 */
	@Override
	public void setProgressBarValue(int value, String s) {
		mainFrame.setProgressBarPosition(value, s);
	}

	/**
	 * enable/disable progressbar
	 */
	public void enableProgressbar(boolean enabled) {
		mainFrame.enableProgressBar(enabled);
	}

	/**
	 * respond to pressing of button "Datumsangaben ndern"
	 */
	public void changeDates() {
		clearTextArea(); // clear log
		Log.outlColor("Aendere Datumsangaben:", LogLevel.useful, Color.blue);

		ChangeDatesWorker worker = new ChangeDatesWorker(this, 0);
		worker.execute();

	}

	/**
	 * respond to button "Scripts starten"
	 */
	public void startScript() {
		clearTextArea(); // clear log
		Log.outlColor("Aendere Datumsangaben:", LogLevel.useful, Color.blue);

		ExecuteScriptsWorker worker = new ExecuteScriptsWorker(this);

		worker.execute();
	}

	/**
	 * respond to button "Test"
	 */
	public void outputDateFields() {

		clearTextArea(); // clear log
		Log.outlColor("Test: Get Metadata from database", LogLevel.useful,
				Color.blue);

		GetMetadataWorker worker = new GetMetadataWorker(this);

		worker.execute();

	}

	public void correctFitnesseScripts() {

		clearTextArea(); // clear log
		Log.outlColor("ndere Fitnessescripts:", LogLevel.useful, Color.blue);
		Log.outl(
				"Transferiere die Skripts zum Basisschuljahr "
						+ Settings.intToSchuljahrString(Settings
								.getFitnesseDestinationSchoolyear()) + ".",
				LogLevel.useful);

		FitnesseScriptCorrector worker = new FitnesseScriptCorrector(this);

		worker.execute();

	}
}
